function val = cont_carryover_seq(num_treatments)
% val = cont_carryover_seq(num_treatments)
%
% returns the first-order (Type-1, Index-1) continuous carryover sequence 
% for <num_treatments> independent treatments (conditions).
%
% The continuous carryover design insures that each treatment appears once
% following each other treatment (including itself), such that first-order
% adaptation effects can be measured for all pairwise combinations of
% treatments. The design is described in detail by:
%
% Nonyane and Theobald (2007). Design Sequences for Sensory Studies: 
%   Achieving Balance for Carry-over and Position Effects. Brit. J. Math. 
%   Stat. Psych. 60:339-349.
%
% Finney and Outhwaite (1956). Serially balanced sequences in bioassay.
%   Proc. Royal Soc. B 493-57. 
%
% Aguirre (2007). Continuous carry-over designs for fMRI. NeuroImage 35:1480-1494.
%
% Sequences returned by this function are taken from Chris Theobald's
% web page at http://www.bioss.ac.uk/staff/cmt/designseq.html.
%
% Sequences are defined for 2 treatments, or any number of treatments between 6 and 34. 


switch(num_treatments)
    case 2
        val = [1 ...
            1 2 2 1];
    case 6
        val = [1 ...
            1  2  3  4  5  6 ...
            6  1  4  3  5  2 ...
            2  5  3  1  6  4 ...
            4  2  6  5  1  3 ...
            3  6  2  4  1  5 ...
            5  4  6  3  2  1];
    case 7
        val = [1 ...
            1  2  3  4  5  6  7 ...
            7  1  3  5  2  4  6 ...
            6  4  7  3  2  1  5 ...
            5  7  2  6  3  1  4 ...
            4  1  6  2  7  5  3 ...
            3  6  5  1  7  4  2 ...
            2  5  4  3  7  6  1];
    case 8
        val = [1 ...
            1  2  3  4  5  6  7  8 ...
            8  1  5  2  4  7  3  6 ...
            6  5  3  1  7  2  8  4 ...
            4  3  2  6  1  8  5  7 ...
            7  6  4  8  2  5  1  3 ...
            3  8  7  5  4  1  6  2 ...
            2  7  1  4  6  8  3  5 ...
            5  8  6  3  7  4  2  1];
    case 9
        val = [1 ...
            1  2  3  4  5  6  7  8  9 ...
            9  1  3  7  4  6  2  5  8 ...
            8  1  6  9  5  7  2  4  3 ...
            3  5  9  2  6  4  1  8  7 ...
            7  6  3  8  5  1  9  4  2 ...
            2  7  9  8  6  5  3  1  4 ...
            4  7  1  5  2  8  3  9  6 ...
            6  8  4  9  3  2  1  7  5 ...
            5  4  8  2  9  7  3  6  1];
    case 10
        val = [1 ...
            1  2  3  4  5  6  7  8  9 10 ...
            10  5  9  1  3  2  4  6  8  7 ...
            7  9  2 10  1  4  3  8  6  5 ...
            5  7  6  9  4  8  2  1 10  3 ...
            3  9  5  8  1  6 10  2  7  4 ...
            4  7  3 10  6  2  5  1  9  8 ...
            8 10  7  2  6  3  1  5  4  9 ...
            9  6  1  8  5  3  7 10  4  2 ...
            2  8  4  1  7  5 10  9  3  6 ...
            6  4 10  8  3  5  2  9  7  1];
    case 11
        val = [1 ...
            1  2  3  4  5  6  7  8  9 10 11 ...
            11  1  5  2  6  3  7 10  8  4  9 ...
            9 11  7  4  6  8 10  5  1  3  2 ...
            2  4  3  5  7 11  6  9  8  1 10 ...
            10  6 11  8  7  9  1  4  2  5  3 ...
            3 11  9  7  5 10  2  1  6  4  8 ...
            8  6  5  9  2 10  3  1 11  4  7 ...
            7  1  8  3 10  9  6  2 11  5  4 ...
            4 10  1  9  3  8  5 11  2  7  6 ...
            6 10  4  1  7  2  8 11  3  9  5 ...
            5  8  2  9  4 11 10  7  3  6  1];
    case 12
        val = [1 ...
            1  2  3  4  5  6  7  8  9 10 11 12 ...
            12  1  3  6  4  7  2  5  8 10  9 11 ...
            11 10 12  9  4  8  5  1  6  2  7  3 ...
            3  5 10  4  9 12  2  8  6 11  1  7 ...
            7  4  2 11  3  8 12  6  1 10  5  9 ...
            9  1  8 11  6 10  7  5 12  3  2  4 ...
            4 10  6 12  7  1  9  8  3 11  5  2 ...
            2  6  5  7  9  3 10  1 11  4 12  8 ...
            8  1 12  5  4 11  2 10  3  9  7  6 ...
            6  9  5 11  8  2  1  4  3  7 12 10 ...
            10  8  7 11  9  2 12  4  6  3  1  5 ...
            5  3 12 11  7 10  2  9  6  8  4  1];
    case 13
        val = [1 ...
            1  2  3  4  5  6  7  8  9 10 11 12 13 ...
            13 11  1  7 10  3 12  2  5  4  8  6  9 ...
            9  7  6 10 12  1  3  8 11 13  5  2  4 ...
            4  9 11 10  5  7 12  6  2  8  1 13  3 ...
            3 13  4 10  9  6  5  8  2  1 12  7 11 ...
            11  2 13  1  6  8 10  4  3  7  5  9 12 ...
            12  8 13  2  6 11  3  5 10  1  9  4  7 ...
            7  9 13  6  3  1 10  8  4  2 12 11  5 ...
            5 11  4 12  3  2  9  1  8  7 13 10  6 ...
            6 12  4 13  8  3  9  5  1 11  7  2 10 ...
            10  2  7  4  6  1  5 13 12  9  3 11  8 ...
            8 12  5  3 10  7  1  4 11  6 13  9  2 ...
            2 11  9  8  5 12 10 13  7  3  6  4  1]; 
    case 14
        val = [1 ...
            1  2  3  4  5  6  7  8  9 10 11 12 13 14 ...
            14  8  4 10  1  5  2  7  6 12  3  9 11 13 ...
            13 12  8 10  6 14  9  1  7 11  5  4  3  2 ...
            2 14 12  1  8  3 10  5  7 13 11  9  4  6 ...
            6 11 10  7  9 14 13  1 12  5  3  8  2  4 ...
            4  9 12 14  6  8 13  2 10  3  5  1 11  7 ...
            7 10  8 11  2  1 13  9  3 12  6  4 14  5 ...
            5  8  6 13  3 14  7  4 11  1  9  2 12 10 ...
            10  9 13  4 12  2  5 11  8  7 14  1  6  3 ...
            3  1  4 13  5  9  6 10 12  7  2  8 14 11 ...
            11 14  2  6  1  3 13  7 12  9  5 10  4  8 ...
            8  5 12  4  2 11  3  7  1 14 10 13  6  9 ...
            9  7  3 11  6  5 14  4  1 10  2 13  8 12 ...
            12 11  4  7  5 13 10 14  3  6  2  9  8  1];
    case 15
        val = [1 ...
            1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 ...
            15 13  1  4 10  5  7  3  6 14  8 11  2  9 12 ...
            12  8 13 15  6  1  3 14  2 10  9  4  7  5 11 ...
            11  1  9  5  3  2  4 12  6 10  8  7 15 14 13 ...
            13 12 10 14  1 15 11  3  7  9  2  5  8  4  6 ...
            6  8  5  2 14 10 12  3  9  7 11 15  1 13  4 ...
            4  9 13  5 10  7 14  6 15  2 12  1 11  8  3 ...
            3 15 12  5  9  6 11 10  1 14  4 13  7  2  8 ...
            8  6  2  7  1 12  9 11  4  3 13 10 15  5 14 ...
            14  3  5 15  4  8  2 11 13  9  1 10  6 12  7 ...
            7  4 11 14  9 15  3 10 13  6  5  1  8 12  2 ...
            2 13  8 10  3 11  9 14 12 15  7  6  4  1  5 ...
            5 12  4 14 11  7 10  2  1  6 13  3  8 15  9 ...
            9  8  1  7 13 11  6  3 12 14  5  4  2 15 10 ...
            10  4 15  8 14  7 12 11  5 13  2  6  9  3  1];
    case 16
        val = [1 ...
            1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 ...
            16  6 13 12 15 14  4  1  7  3  2  5  8 10  9 11 ...
            11  4 15  1  6 12 16  7  2  8  3 10  5  9 14 13 ...
            13  4  9  5 16  1 14  6  8  7 10 12 11  2 15  3 ...
            3  5  7  4 13 15  2 12 10  6 16  8 11 14  1  9 ...
            9 12 14  2 10  7 13  8  4  3 11  1 16 15  6  5 ...
            5 10 13  9  7 12  2 11  8  6 14 16  3  1 15  4 ...
            4  6 11 16  2  7  5 14  3 13 10 15 12  9  1  8 ...
            8 16  9 15 10 14  7 11  3  6  1 13  5 12  4  2 ...
            2  4  8 12  1  3 14  5 13 16 11 15  7  9  6 10 ...
            10  8 13  1  5 11  6  9  2 14 12  3 16  4  7 15 ...
            15 11  9 16 10  1  4 12  5  2 13  7  6  3  8 14 ...
            14  8 15  9  3 12  7  1 11  5  4 10  2 16 13  6 ...
            6  2  1 10 16 12  8  5 15 13  3  9  4 14 11  7 ...
            7 14  9 13 11 10  3 15  8  2  6  4 16  5  1 12 ...
            12  6 15  5  3  7 16 14 10  4 11 13  2  9  8  1]; 
    case 17
        val = [1 ...
            1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 ...
            17 10 12 15 11  8 13 16  6  9  1  7 14  2  4  3  5 ...
            5  9 14 11 13 12 16 10 17  8 15  1  3  2  6  4  7 ...
            7 12  2 10  3 13 15  6  5 11 14 16  9 17  1  4  8 ...
            8 14 17  6  2 15  9 16  3 11  1 12  4 10  5  7 13 ...
            13  2  5 15 14  3 16  1  6  8  7  4 17 12  9 11 10 ...
            10  4  9  2  7  6 17  3  8 11 15 12 14 13  1  5 16 ...
            16 13  9  3  6  1 17  7 15  4  2 12 11  5  8 10 14 ...
            14  1  8 16  4  6 10  7 17 11  3  9 15  2 13  5 12 ...
            12 10 15 17  2  1 11 16  5  3 14  6 13  7  9  8  4 ...
            4 11  7 16  8  1  9 12  5 13  6 14 10  2 17 15  3 ...
            3  7  5 10  1 13  4 14  9  6 11  2  8 12 17 16 15 ...
            15  5 14 12  1 16  2  9  7 10  8  3 17  4 13 11  6 ...
            6 12  3 15 13 10 16  7 11  9  4  1 14  8 17  5  2 ...
            2 11 17 13  8  5  4 12  6 16 14  7  3  1 15 10  9 ...
            9  5  1 10 13 17 14  4 15  8  6  3 12  7  2 16 11 ...
            11  4 16 12  8  2 14  5 17  9 13  3 10  6 15  7  1];
    case 18
        val = [ 1 ...
            1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 ...
            18  8  2 10 13 15 11  1 17  6 12  9  3  5  4 14  7 16 ...
            16 10 12 11  2  4  6  1  3 17  9  5  7 13 18 14  8 15 ...
            15 18  5 11 16 14  9 17 10  1  6 13  2 12  3  7  4  8 ...
            8  6  3  1  9  7  5 18  4 10 14 11 13 12  2 16 15 17 ...
            17 11 14 16 12 10 18  1  4  2  5 15  3 13  6  9  8  7 ...
            7  9 18 17 12  8 16 13  3 14  4 15  1  5  2  6 11 10 ...
            10  3 15  2  7 11  6  4 16  9 12 17 14 18 13  1  8  5 ...
            5  9 11 17  7 14 10  6  8  3 12 15 13 16  4  1 18  2 ...
            2 13 17  8 10 15 12 14  1 16 18  7  6  5  3  9  4 11 ...
            11 15 14 12  4 13  5  8 17  1 10  7 18  3 16  6  2  9 ...
            9  1  7  2  8  4 12 16 11 18  6 14  3 10 17 15  5 13 ...
            13  4 17 16  3 18  9 14  2  1 11  5 10  8 12  7 15  6 ...
            6 15  4 18 10 16  5 12  1 13  7 17  3  8 11  9  2 14 ...
            14  5 16  1 12  6 17  2 15  7 10  4  9 13  8 18 11  3 ...
            3  6 18 16  8  1 14 13  9 15 10  2 11  7 12  5 17  4 ...
            4  7  3 11  8 14 17 13 10  5  1 15  9  6 16  2 18 12 ...
            12 18 15  8 13 11  4  3  2 17  5 14  6 10  9 16  7  1];
    case 19
        val = [1 ...
            1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 19 ...
            19 17 14 18 12 15 10  7  5 16  1  4  8  2  6  3 13  9 11 ...
            11  9 12 14  8 15 13  7 16 19 10 17  1  3 18  4  6  2  5 ...
            5 12  1  8 13 10 15 11 14 19 16 18 17  2  4  3  9  7  6 ...
            6  4 18  9 19 14 12 11 10 13 15 17 16  2  1  5  7  3  8 ...
            8 16  3  7  1 17 10  9  5 13 11 18  2 12 19 15  6 14  4 ...
            4 12 16 11  3 15  2 17  5 18 14  6  8 19  7  9 13  1 10 ...
            10  1 11  4 16  7 19 18  3  2 13 12  5 17  6  9 15  8 14 ...
            14 13  4  9 16 15 12  2 18  7 10  5 11 17  8  6  1 19  3 ...
            3  6  5  8 17  7 11 13 18 16  9 14 10 19  1 12  4  2 15 ...
            15  9  8  1  6 11 19  2  7 12 17 13 16  4 14  3  5 10 18 ...
            18  1 15 19  5  2 10 14  7  4 11  6 13  3 16  8 12  9 17 ...
            17 19  4 13  2  8 10  6 15  7 14 16  5  3 12 18 11  1  9 ...
            9 18  5 14 11  2 16  6 10  4  7 15  1 13 19  8  3 17 12 ...
            12  3 14  9  2 19  6 16 10  8  4 15  5  1 18 13 17 11  7 ...
            7 17  3 19  9  1 16 13  5  4 10 12  6 18  8 11 15 14  2 ...
            2  9  6 12 10 16 14 17  4  1  7 18 15  3 11  8  5 19 13 ...
            13  6 17 15 18 10  3  1 14  5  9  4 19 12  8  7  2 11 16 ...
            16 12  7 13  8 18  6 19 11  5 15  4 17  9  3 10  2 14  1];
    case 20
        val = [1 ...
            1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 19 20 ...
            20 19  8 16  9  2  5 17  1 11 18  3 15  6 14  4  7 10 13 12 ...
            12  1  9 16 15 17  5 11  8 20 13 18  2 10  3 14  6 19  7  4 ...
            4  6  1 19  9  3  7  5 20 10  8  2 11 13 15 12 18 14 17 16 ...
            16 18  4 14  7  3  8 15 19 17 11 20  1  5  2  6 13  9 12 10 ...
            10 15  7 12 14 18  8  4 20 11 16 19  1  3  9  5 13  2 17  6 ...
            6  5  7 15 18  9 20  3 19  2 12  8 10  1 13 11 14 16  4 17 ...
            17 20  2  7  6  3 10  5 14 19 11  1  4  8 12 15  9 18 16 13 ...
            13 16 10 17  7  9  6 11  5 18 15 20 12 19 14  2  4  1  8  3 ...
            3 11 10 12 16  7 13 17 14 20  4 15  8  5 19 18  1  6  2  9 ...
            9 13  3 18  5 15 14  1  7 16 20 17  4 10 19 12  6  8 11  2 ...
            2  8  6 12  7 11  4  9 14 13 19  3 16  5  1 10 20 18 17 15 ...
            15  1 12  9 17 19 10  2 13  4 11  6 16 14  5  3 20  8  7 18 ...
            18 20  5 12  3 13  1 17  2 14  8 19  4 16  6  9 11 15 10  7 ...
            7 14  9 15  2 16  1 18 10  4 19  5  8 13 20  6 17  3 12 11 ...
            11  9 19  6  4 12 17 13  5 10 16  8 18  7  1 15  3  2 20 14 ...
            14 10 18 11  3  1 16  2 19 15  4 13  6 20  7 17 12  5  9  8 ...
            8 17  9  1 20 15 13  7  2 18  6 10 14 11 19 16 12  4  3  5 ...
            5  4  2 15 11 17  8  1 14 12 20 16  3  6 18 13 10  9  7 19 ...
            19 13  8 14  3 17 10  6 15  5 16 11  7 20  9  4 18 12  2  1];
    case 21
        val = [1 ...
            1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 19 20 21 ...
            21  9  3  5  7 19 12 18  2 10 13 20 11  8 14  1 15  4  6 17 16 ...
            16 10  3 21 11 15 17  4  7  5  1  6 14  8 13  2 20  9 12 19 18 ...
            18 12 11  6 13  4  8 19 17 20  2 21 10 14 16  1  5  3  7  9 15 ...
            15 12 16 18  6  9 13  1 19 14 20  3 11 17  7 21  8  5  2  4 10 ...
            10  9  1 11  7 12 20  8 15 13 16 14 17 19 21 18  3  2  6  5  4 ...
            4  9 14  3 17 12  5  8  6 20 18  1  7 15 10 16 13 11 21 19  2 ...
            2  1 12 17  5 18 16 19  9 20  4 14  6  8 21 15  3 13 10  7 11 ...
            11 20 16 21 13 15 14  7  1 17  2  8 10 18  4  3  6 12  9  5 19 ...
            19 10  8  1 13  5  9  6 18 20  7  2 11  3 12  4 16 15 21 17 14 ...
            14 18 13  3 15  2  5 16  6 10 17 21 20 12  1  4 19  8 11  9  7 ...
            7 18 21  2 17 15  8  3  9  4  1 10 12 14 19  5 13  6 11 16 20 ...
            20 19  1  8 16  7 17  9 21  3 14  2 12  6 15 11 10  4 13 18  5 ...
            5 12 10 15  6 16  2 18  7 13 17 11 19  3  8 20 14  4 21  1  9 ...
            9 16 11 18 14 10 19  4 15  5 21  6  1 20 17  8  2 13 12  7  3 ...
            3 16  4 11  1 14  5 17  6 19 13 21  7 20 10  2 15 18  9  8 12 ...
            12 21 14  9 11 13 19  6  2  7  4 17  3 18 10 20  5 15  1 16  8 ...
            8  4 20 15 19  7  6 21  5 11 14 13  9  2 16 12  3 10  1 18 17 ...
            17  1 21  4  2 19 16  3 20 13  8  7 14 12 15  9 18 11  5 10  6 ...
            6  4 12  8 18 15  7 10  5 20  1  3 19 11  2 14 21 16  9 17 13 ...
            13  7 16  5 14 11  4 18  8 17 10 21 12  2  9 19 15 20  6  3  1];
    case 22
        val = [1 ...
            1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 19 20 21 22 ...
            22 18  6  1 19  4  7  2 14  8 20  9 21 10 15 17  3  5 11 13 12 16 ...
            16 15 10  1 22  6  5  9 19 21 17  2  4 20 11  7  3  8 12 14 18 13 ...
            13 18 21 16 22  7  4  1 20 14  2  5  8  3  9  6 17 10 12 15 11 19 ...
            19  5  1 11 14 13 15 18  8 10  2  6  9 16 20  7 12 17 21  3 22  4 ...
            4  3 11  8 13  5  7  6  2 18 15 12 21  9 14 16 10 19 22 20  1 17 ...
            17 15  6 10  3 12 18 11 20 13  8 14  4 19 16  1 21  2  7 22  9  5 ...
            5 12 10 13  4  8  6  3 14 17 22 11 15 19  9  1  7 16 18 20  2 21 ...
            21 14  7 15  8  2 22 12  5  3 20 16  4  9 11  6 13 17 19 18  1 10 ...
            10  9 20  3 13 11 16 14 12 19  1  4 15  7 17  5 18 22  2  8 21  6 ...
            6 20 10 21  7 19 11  3 15  9 17  1  8  4 18 12 22 13  2 16  5 14 ...
            14  9 22 21 18  7 13 16  2  1  3 19 17 20  5  4  6 12 11 10  8 15 ...
            15 20 17  6 21  4 10 22 19  7 14  1 13  3  2  9  8 11 18  5 16 12 ...
            12  3 17  9  7 11 22 14  5 19  2 10 16 21  1 15  4 13 20  6  8 18 ...
            18  9  2 11 17  4 12 20  8 22 10 14 19  3 21 15  1  5 13  6 16  7 ...
            7 20 22  1 18 10 17  8  5 15 13 21 19 12  4 16  9  3  6 14 11  2 ...
            2 12  1  6 18 16  8 19 10 20  4 11 21  5 17  7  9 13 22 15 14  3 ...
            3 16 19  6 11  1  9 12  2 13  7  5 20 15 22 17 14 10 18  4 21  8 ...
            8  1 12  6 22 16 11  5 10  4 14 21 20 19 15  3  7 18  2 17 13  9 ...
            9 18 17 12  8 16  6 19 13 10  7 21 11  4  2 15  5 22  3  1 14 20 ...
            20 12  9 15 21 13  1 16  3 18 14 22  5  2 19  8  7 10  6  4 17 11 ...
            11  9  4 22  8 17 16 13 19 14  6 15  2 20 18  3 10  5 21 12  7  1];
    case 23
        val = [1 ...
            1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 19 20 21 22 23 ...
            23  4 12  9 16 18 10 15 19 22  8  2  1 14  6 11 20  7 21 17 13  3  5 ...
            5  7  6  3  2 15 13  4  1 16 21  8 18 22 10 17  9 11 14 12 19 23 20 ...
            20 11 22  4  2  5 17 23 19  6 14 21 18  1  3 15  7 13  8 12 10 16  9 ...
            9  3 16 19 21 11 15 22 12 14 10  8 13 20 23  5  1  4  6 17  2 18  7 ...
            7 14  5 20 12  6  8 19 11  3  9 13 10  1 15 17  4 22  2 21 16 23 18 ...
            18  8 10 12 16  2 11 21 19  1 17 20 22 13  6  9 23  3  7 15  5  4 14 ...
            14 17 21 12 18 23 11  1  8 20  2 13 19 15  3 22 16  4  7  5  9  6 10 ...
            10 23  6 15 18 12 20 19  2 14  3  8 11  7  1 13 21  4  9 17 16  5 22 ...
            22 14 20  1 18 17 10  2 19  7  3 21  5  8  6 23  9 12  4 11 16 13 15 ...
            15  9 14  4  3 11  5 10  6 16 20 17  8 21 23 13 18  2  7 22  1 19 12 ...
            12 11 23 21  1  6 13 16 10 20 14 18  9  2  4 15  8  5  3 17 22  7 19 ...
            19 10 13  5  2 22 15 11  4 16  1  7  9 18 20  3  6 12 21 14 23  8 17 ...
            17  7 20  5 15  4 18  3 10  9 19  8 22  6  1 21 13  2 12 23 14 16 11 ...
            11 10  4  8  7 12 22 17 19  3 13  9 20  6 18 14  1  5 21 15 23  2 16 ...
            16  6 19  9 15  1 10 18 13 22 20  4 21  2 23  7 11 17  3 12  5 14  8 ...
            8 23 10  7 16 22 21 20 18  4 13  1  9  5 19 14 11  6  2 17 15 12  3 ...
            3 14 22 19 13 17  1 20  8  4 23 12  2  6  5 11  9  7 18 16 15 10 21 ...
            21  3 18  5 16  7  2  8  1 23 15 14  9 22 11 13 12 17  6 20 10 19  4 ...
            4 17  5 12  7 23 16  3 19 18  6 22  9  1 11  8 15 21 10 14  2 20 13 ...
            13 23 22  3  1 12 15 20  9  4 19 16  8 14  7 17 11  2 10  5 18 21  6 ...
            6 21  9  8 16 12  1 22 18 11 19  5 23 17 14 13  7  4 10  3 20 15  2 ...
            2  9 21  7 10 22  5 13 11 18 15  6  4 20 16 14 19 17 12  8  3 23  1];
    case 24
        val = [ 1 ...
            1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 ...
            24 18  6 19  2 23  1  4  7 11  3  5  8 10  9 12 16 21 17 13 15 22 20 14 ...
            14 17 19 16  2 22 21 20 13 23  3 18  4 24  5  1  6  8  7  9 11 10 12 15 ...
            15 18  7  6 16  9  8 11 13  3 23  5 24 19 10  2  4 12 14 20 17 21  1 22 ...
            22  7  2  1  3 11 17 20  4 19 12 21  5 18  9  6 23  8 13 16 24 15 14 10 ...
            10 21  8  2 11  9 20 18 22 12 17 23 13  6  5 19 14 24  1 15  3  7  4 16 ...
            16  1 19 21 13 20 23 12  3  8 24 22  2  5 10 17  4  6  9  7 14 18 15 11 ...
            11 18  8  4  1 12 19 13 24 14  6 10 15 21  2 16  3  9 17  7 20 22  5 23 ...
            23 16 10 24  7 21 19 22  6 11 14  4  2  8  5  3 12 18 20 15  1  9 13 17 ...
            17 14 19 15  5  7  3  6 22 24 16 23 18 21  4  8 12  9  1 10 20 11  2 13 ...
            13  1 20 19  9 21  6 12 22  3 17 24 23  2  7 16  8 14  5 15 10 18 11  4 ...
            4 15 12 20 16 18  3 10 19 11 21  9 14 22  1  5 13  2 17  6 24  8 23  7 ...
            7  5  9  2 24 17  8 15  4 10 13 18 12 23 19  1 14  3 16 22 11 20  6 21 ...
            21  3 13  8 16 20 24  9 22 14 23  4 11  5  2 10  1 17 12  6 15 19  7 18 ...
            18 10 14 11 15 20  1  7 12 24 21 23 17 22  4  3  2 19  8  6 13  9 16  5 ...
            5  4  9 18 23 10 22  8 17 11  1 21  7 13 12  2  6 14 16 15 24 20  3 19 ...
            19 17  5 11  6  4 18 24 10  3 14 21 12  1  8 20  2 15 13 22 16  7 23  9 ...
            9 15  6 17  1 23 20 12  4 13  5 14  2 18 16 11  7 22 19 24  3 21 10  8 ...
            8 19 18 14  1 11 22  9 23 21 15  7 10  6 20  5 16 13  4 17  3 24  2 12 ...
            12  5 17  2 14 13 19 23 15  9  3 20  7 24 11  8 22 10  4 21 18  1 16  6 ...
            6 18 13 10 23 22 15  2 20  8  1 24 12  7 17 16  4 14  9  5 21 11 19  3 ...
            3 22 13 11 24  4 23 14  7  1 18 17 15  8 21 16 12 10  5 20  9 19  6  2 ...
            2  9 24  6  3  1 13 21 14 12  8 18  5 22 17 10  7 15 23 11 16 19  4 20 ...
            20 10 16 14  8  3 15 17  9  4 22 18  2 21 24 13  7 19  5 12 11 23  6  1];
    case 25
        val = [ 1 ...
            1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 ...
            25  6  4 18 15 13  7 16  2 21 14  3 19 22 17 20  5 23  1  8 24  9 11 10 12 ...
            12 14 20 22  9 23 13 10 24 11  7 15 25  1 19 18  8 17  2  4  3 16  5 21  6 ...
            6  8 10 13  1  5  2 22 19 16 18  3 20  9  4 11 14 25 12 15  7 17 23 21 24 ...
            24 19  7  5 20  8 25  9 12 10 14 13  4 15  1 22  2 11  3  6 16 21 23 18 17 ...
            17 16  6 21  7 18  1 11 19 23  8 20 24 10 22 25  5  3  2  9 13 15 12  4 14 ...
            14 16  3  5 24 12 11  1  9 17  4 20  7 10 15 21 25 13 23 19  8  6 22 18  2 ...
            2 19 17 21  4 23 25 20  6 13  3  1  7  9  5  8 22 10 16 11 15 14 12 24 18 ...
            18 21  1 24 15 17 25  8  5 22 16 23 20 10  2 14 19  9  3  7  4 12  6 11 13 ...
            13  5  4 21  8  1 16  7 11 17 22 15 18 23 14  2 20 12 19  3 25 24  6 10  9 ...
            9 14 23  2  5  7 12 16  8 21 15 10 20 25  3 17 11 18 22  4 13 24  1  6 19 ...
            19  5 12 22  7 20 11 21 16  4  8 13  9  1 25 14 18 24  2 23 15  6 17  3 10 ...
            10 17 24 13  6 12 23 16  9 18  4  7 14 21  2 25 11  8 15 22  5 19  1 20  3 ...
            3  8 18 20 19 15  2 24 21  5 11  4 16 22  1 23  6 14 10 25 17 13 12  9  7 ...
            7 13 19 21  3  9  8 23 22 12 17 15 20 14 24  4  6 18 11 16 25  2  1 10  5 ...
            5  9  6 15 19 24  7 23 11  2  8  3 12 25 18 10  1 21 13 16 20  4 17 14 22 ...
            22 13 25  7 19  2 10  3 23  4  9 24 16  1 12 21 11  5 18 14 17  6 20 15  8 ...
            8  2 16 10 18  5 13 22 20 17  1 14  4 19 11  6  9 15 24  3 21 12  7 25 23 ...
            23 10  6 24  5 14  1 13 18 12 20 16 15  3 22  8 11  9 19 25  4  2 17  7 21 ...
            21 19  4 22 11 20 23  9 25 15  5 10  7  2 12  3 24 14  6  1 18 13 17  8 16 ...
            16 12 18  6  3 11 22 14  5 25 19 13 20  1 15 23 17 10 21  9  2  7 24  8  4 ...
            4 24 17  5  1  3 13  8  7 22  6 25 21 10 23 12  2 15  9 20 18 16 19 14 11 ...
            11 25 22 24 23  3 18  7  1 17  9 16 14  8 12  5 15  4 10 19  6  2 13 21 20 ...
            20  2 18  9 21 17 19 12  1  4 25 10  8 14  7  6 23  5 16 13 11 24 22  3 15 ...
            15 11 23  7  3 14  9 22 21 18 25 16 24 20 13  2  6  5 17 12  8 19 10  4  1];
    case 26
        val = [ 1 ...
            1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 ...
            26  5 23 10 16  1 17  9  6 15 18 20  7 11  3 24 12 14  8 13 19 21 25 22  2  4 ...
            4  6 11 14  9  8 19  5 24 18 21  7 10 20 12  2  1  3 13 15 17 22 16 23 26 25 ...
            25  7 13  1 19 26  4 20  2  5  8  3 21  9 14  6 12 10 15 11 16 18 17 23 22 24 ...
            24  8 18 22  1 20 11 17 25 15 10  3  9 19  4 23 13 26 21 14  2 16  6  5  7 12 ...
            12  3  6 16  9  7 24 23 14 13  2 15  4 25 11  8  1  5 10 18 26 17 19 22 21 20 ...
            20  5 16 10  7 14 11  6 13  9 17 24 15 21  3  8 12 25 18 23 19  1 22  4 26  2 ...
            2 12  4 15  5 22 14  7 17  3  1 18 16  8 20  9 11 10 19 23 25 21 13 24 26  6 ...
            6  2 26 22 12 15 23  1 24 14  3 25  4 11  7  9  5 17  8 16 13 20 10 21 19 18 ...
            18 14 21 26 23 20 19  6 17 15 22  5 25  1  4  3  7  2 11 24  9 12  8 10 13 16 ...
            16 14 10 22 20 13  3 26 24 11  4 19 25 17 12 21 23 18  1  6  8  2  7  5 15  9 ...
            9  2 10 17  4 21 18 13 22  3 19  7 16 20  8 23  5 26  1 11 15 12 24  6 25 14 ...
            14 12  6 19 10 23 16 15 13  4 17 26 11 22 18 24  1  7  3 20 25  2  8  5  9 21 ...
            21  6 14 16 22 17  5  1 23  9 13 10 12 26 15 20  3 18 25 19  2 24  7  4  8 11 ...
            11 13  5  2 22 15 14 17  6 23 12  1  8 21 16 24  4  7 18  3 10 25  9 20 26 19 ...
            19  8 15 24 10  2 25 12  9 26 20  1 13 11  5 21 17  7 22  6 18  4 16  3 14 23 ...
            23 11 20  4  9 22 19 12  5  3 17 14 18  2 13  6 21 24 16 25  8  7 26 10  1 15 ...
            15  8 14 19 16 26  3 12  7  1 25 13 17 10  9  4 18  5 11 21  2 20 23  6 24 22 ...
            22 25 23 17 20 24 13  7 15 19  3  2  6 26  8  4  1 21 11  9 16 12 18 10 14  5 ...
            5 12 11 23 21  1 26 13 18 15 25  3 22  7  6  9 24 17  2 19 14 20 16  4 10  8 ...
            8 17 13 21 12 16 19  9 23  2 14 22 26 18  6 20 15  7 25  5  4 24  3 11  1 10 ...
            10 24 19 11 18  7 20 14  4 22  8  6  3 23 15 26 12 17  1  9 25 16  2 21  5 13 ...
            13  8 22  9  1 14 24  2 17 21 10 26 16  5 18 11 25 20  6  4 12 23  7 19 15  3 ...
            3 16 21  4 14 25 24 20 17 11  2 18  9 15  6  1 12 22 10  5 19 13 23  8 26  7 ...
            7 21  8 24  5 14 26  9 18 12 20 22 13 25  6 10  4  2 23  3 15  1 16 11 19 17 ...
            17 16  7 23  4 13 12 19 24 21 15  2  9  3  5 20 18  8 25 10  6 22 11 26 14  1];
    case 27
        val = [ 1 ...
            1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 ...
            27  2 11 13 25  3  1 22  4 19  8 12 10 23  5 14 24 26 20  6  9 16  7 15 21 18 17 ...
            17 19  4 15 14  7  9 18 12 22 26 21 16 20  8 10  3 11  2 13 23  1  6 24 27 25  5 ...
            5 27 22 19 18 15 23  2  4  1  3 14 20 24  7  6 25  8 11  9 26 16 21 10 12 17 13 ...
            13  6 26  1 27  3 10 20 14 22  5  8 23  9 11  7 21 12 16 24  2 15 25 19 17  4 18 ...
            18 25 15 20  5 17  3  8 26  4 23 10 13 19 12 21 24 22 14 11 27 16  9  1  7  2  6 ...
            6 20 22  7 18 27  4  8 25 13 24  5 23 26  3  2  9 12 11 14 16  1 21 15 17 10 19 ...
            19 13  2 22 24  3 20 12 23  4  9 21 25 27  1 26 10  5  7 11 17 15  6  8 14 18 16 ...
            16  4 20  7 26  8 15  9 23 11  3  5 13  1 12  6 14 17 21 19 22 18 10 25  2 27 24 ...
            24 19 16  3 17  6 11 18 13 21 26 12 15  5  2 14 25 10 22  8 20 27  9  4  7  1 23 ...
            23 19 11 20  3  6 21 27 26 24 16 10 17 22 12  4 25  1  8 13 15  2 18  5  9  7 14 ...
            14 23  7  3 19 21 13  4 12 24 11 10 16  2 25  6 17 20 26 22 15 18  8 27  5  1  9 ...
            9  2 16 26 11  4 27 10  6  1 19 25 17 23 12  3 21  5 18 14  8 22 20 13  7 24 15 ...
            15 26  9 24 10 14 27  6  2 12 19 23 13 18  1 16  5 20 17  7 22 21  3 25  4 11  8 ...
            8 16 14 12 18  2 17 26 25 21 23 20 15 10 24  1  4  3 27  7 13  5 11 19  9  6 22 ...
            22  3  7 23  8 21 20 16 15 24 17  2 19 14 26 13  9 25 11  1 10 18  6 27 12  5  4 ...
            4  6  3  9 27 14  5 15  7 10 21 17  1 11 16  8 18 26 23 22 13 12 25 24 20 19  2 ...
            2  5 19 10 15 11 21  1 24  4 22 16  6 12 26 14  9 17 25 18 20 23  3 13 27  8  7 ...
            7 25 14  1 17 27 11  5 10 26 15 12  2 23  6 16 13 20 18 22  9 19  3 24  8  4 21 ...
            21 11 15 13  8 24 23 14  2 10 27 17  5 22 25 16 12  1 18  4 26  6 19  7 20  9  3 ...
            3 22  6  4 10  1 13 16 18  7 12  9 20 11 23 21  2 26 17 24 14 19 27 15  8  5 25 ...
            25 12 14 13 22 27 18  9  5 16 11 24 21  4 17  8  1 20 10  2  7 19  6 23 15  3 26 ...
            26 19  1 25  7 27 13 17  9  8  2 21 14  3 15 22 10  4 16 23 18 11  6  5 24 12 20 ...
            20 25  9 13 26 18  3 23 16 19 24  6 15 27 21  7  4 14 10  8 17 11 22  2  1  5 12 ...
            12  8 19  5 26  2 20  1 15  4 13  3 18 24  9 14 21  6 10  7 17 16 27 23 25 22 11 ...
            11 25 23 17 12 27 20  2  8  6 18 21  9 15 19 26  7  5  3 16 22  1 14  4 24 13 10 ...
            10  9 22 17 14  6 13 11 26  5 21  8  3 12  7 16 25 20  4  2 24 18 23 27 19 15  1];
    case 28
        val = [ 1 ...
            1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 ...
            28  6  9  4 25  3 15 14  5  7 13 19 22  8 10 12 11 20 23 17  1 24 16 18 26  2 27 21 ...
            21 24 13 15 17  5 27 20 11 19 25 28 10  6  4 23 18  1 16 22 26  7  2  8 12  3  9 14 ...
            14 20 22  5 15 10 24 21 23 25 18 27  6  3 16 13 17  7 26 28  1 11  2  9  8 19  4 12 ...
            12 14 23 16 24 15 13  7 27 17 25 20 18 21 26 19 28 22  1  3 10  2  4  6  5  8 11  9 ...
            9 21 18 20  8 16 26 17 22 24  2 10 13 11 14  1  7 15 25 27  3  5 12 19 23  6 28  4 ...
            4 24  3 25 16 19  9 15 11  5 26  1  6 14 21 20 10  7 12 17 23 22 18 28  8 13 27  2 ...
            2  1  8 22  3 12  9 20 24 23 19 18 10  4  7  5 28 21  6 25 11 13 16 14 17 15 27 26 ...
            26 14  7 22 25 21 11 23 27 18 17  6 19  8  5  1 20  2 12 10 16 28  3 13  9 24  4 15 ...
            15 21  3  6 24  5  9 13 18 23  2 22 27 25  1 10  8 14 11 26 20 28  7 16 12  4 17 19 ...
            19  2 11 28 14  6 15 18  3 21  5 10 25 17 13 20 26 22 16 23  1 27  7  4  9 12 24  8 ...
            8 15 24 22  9 25 10 14 13  1  4 26  5  2 16  3 27 19 17 20  6 11 21 28 18 12 23  7 ...
            7 28 11 18  8 17 21 14 24 19 26 25  4 20 27  1 22 12  2 15 23  9  5 13  6 16 10  3 ...
            3  2 13 12 20  5 16 15 19 27 14  4 18  6  1  9 28 23 21 25  8  7 10 17 24 26 11 22 ...
            22 17 26  8 18 14 19  7  3 11 15  1 12 21 16 20 25  2 28  5  4 10  9 23 13 24  6 27 ...
            27  8 26  6 10 28  9 16 21 19 15  2 23  4 22 20  1 17  3  7 11 24 12  5 25 14 18 13 ...
            13 21 10  5 11  6 22 28 20  3 23 26 15  9 27 16 25  7  1 14  2 18  4 19 12  8 24 17 ...
            17 27 23 12 18 15 20  9  6 13  2 26 16 11 10 21  4 28 19 14  3  8 25 22  7 24  1  5 ...
            5 14  8  1 21 27  9 22  2 17 10 23 20  4 11  3 18 24  7 19  6 26 13 25 12 15 28 16 ...
            16 27 24 28  2 21  7 23  3 19  1 13  8  4 14  9 17 11 25 15 22  6 12 26 10 18  5 20 ...
            20  7 25 19 24 10 15 26  3 14 12 27  4  1 18 16  2  5 17 28 13 22 21  9 11  8  6 23 ...
            23 28 12 16  1 26  4 13  3 17  2 25  5 24 14 22 10 20 15  6 18  7  9 19 21  8 27 11 ...
            11  4  8 20 12  7  6  2 24  9 18 25 23 14 26 21 17 16  5 22 13 28 27 15  3  1 19 10 ...
            10 19  3 20 13  4 21 12 25 24 18  2  7 17  8 28 26  9  1 15  5 23 11 27 22 14 16  6 ...
            6 17 12 22  4 16  7 18  9 26 23 10 27 13  5 21  2 14 28 15  8  3 24 20 19 11  1 25 ...
            25  6  8 21 13 23 15  7 14 27 10 26 12  1 28 24 11 16  9  2 20 17  4  3 22 19  5 18 ...
            18 22 11  7 20 16  4 27  5 19 13 10  1 23  8  2  6 21 15 12 28 17 14 25  9  3 26 24 ...
            24 27 12  6 20 14 10 22 15  4  2 19 16  8 23  5  3 28 25 13 26 18 11 17  9  7 21  1];
    case 29
        val = [ 1 ...
            1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 ...
            29  7 28 12  3 19  1  4  6 13 15 26  8 14 20 22 16 24 11 27 17  5  2 21 10  9 18 23 25 ...
            25 24  5 29 28 10  1 21 17 23 11 19  2 16  3 22  4 20 13 26  6  9  8 27  7 12 14 18 15 ...
            15 11  8  7  3 27 25 17 10 21 18 14 16  5 28  9 24 12 19  4 13 20 23 22  2  6 29  1 26 ...
            26  1  8 10  7 19 15  3 16 29 17 20  9 11 22  6 21 12 27 13  4 14  5 18 24  2 25 28 23 ...
            23 26  4 27  5 25 29  9 17 14 24 28  1  3  6 22 13 18  7  2  8 11 10 19 12 15 20 16 21 ...
            21 11 17  3 20 18 16  6 14 27 19  9 22 15 23 28  4 29  5 10 24  7 13  1 12 25  8 26  2 ...
            2  7 14 10  6  4 23  5 17 21 13 16 15 25 11  9 20 24 18  8 19 22  1 27 26 28  3 29 12 ...
            12  9 21 14 22 17  4 15  5  7 20 19 24 13 23 29 25 16  1  6  2 18  3 26 10  8 28 27 11 ...
            11 13 17 19  5 20  6 18 27 12 16 28 21 15 14 26 29  3  1 22 24  4 25 23  8  2  9  7 10 ...
            10  4 26  3 23 27 24  6  8 20 25 21  5 16 22 11 28  2 29 13  7  9  1 18 12 17 15 19 14 ...
            14  6 12 10 29 19 13 21 16 25  7 22 20 26 15  8 23  9  4 11  2  5 24 27  3 17  1 28 18 ...
            18 29 10  5  8 21 19  7 15  6 11 25  9 12  4 16  2 28 13 27  1 24 14 17 22 26 23 20  3 ...
            3 18 13  2  1 25 10 12  5 11 29 20 14  7  6 23  4 17  8 22  9 15 21 24 19 28 26 16 27 ...
            27 18 28 19 29 22  7  4  9 23 15 12  1 11  3 24 10 16  8  5 13 25  2 17  6 26 14 21 20 ...
            20 10  3 21  9 19  6 15  2  4 28 22 25 27 23  7  5 12 26 24 16 11  1 29 14  8 18 17 13 ...
            13 19  8 24 29 15  1  9 25 18 20 27  2 26  5 21 23  6 28  7 11 16  4  3 14 12 22 10 17 ...
            17 24 26 20  2 10 18  4  8 25  1 19 27  6  3 13  5  9 14 23 12 28 11 15 22 21 29 16  7 ...
            7 16 26 21 27  8 29  2 11 20 28 14 19 17 25  3 12 23 10 13 22 18  1 15  4 24  9  6  5 ...
            5 26  7 27 14  1 17  2 12 21 25 13 24  3 15 28  6 10 20 29  4 19 11 23 18 22  8 16  9 ...
            9 26 11 14 25 15  7  1 10 22  3  2 13 12 29  8  4 18  5 19 21  6 27 20 17 28 24 23 16 ...
            16 12 18  9 27  4 21  7 17 29 11 24  8 13  6  1 20  5 14 28 15 10  2 23  3 25 19 26 22 ...
            22  5 15 24 17 26  9 16 10 14  2 19 18 25  4  1 23 13 11  6 20  8 12  7 29 27 21  3 28 ...
            28  5 27 22 12 20  1  7 18 21  2 14 29 23 17 16 13  8 15  9  3 11  4 10 26 19 25  6 24 ...
            24 20 11 18 26 17 12  2 27  9 28 25 22 29  6 16 14  4  7 21  1  5 23 19 10 15 13  3  8 ...
            8  1 13 28 16 18  2 15 17  9 29 21 26 25 20 12  6 19  3  7 24 22 27 10 23 14 11  5  4 ...
            4 22 19 16 23  2 24  1 14 13  9  5  3 10 28 20  7 25 12 11 21  8 17 27 15 29 26 18  6 ...
            6 25 14  9  2 20 15 18 11 26 13 10 27 29 24 21  4 12  8  3  5 22 28 17  7 23  1 16 19 ...
            19 23 21 28  8  6 17 11  7 26 12 24 15 27 16 20  4  2 22 14  3  9 13 29 18 10 25  5  1];
    case 30
        val = [ 1 ...
            1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 ...
            30  1 23 14 28 12 15 20 13 10 29 18  7 17  8 11  9 19  4 21 24 16  2 22  3  5 27 25  6 26 ...
            26 21 23 27 15 17 28 13 11 14 24 18 22 25  4 16 10  6 12 19  1 29  2 30  9 20  3  7  5  8 ...
            8 18 27 10  1 19 11 29 20 30 12  9 28 23 25 22  6 13 24 14 26 15 21  2  4  3 16  5 17  7 ...
            7  6 30 15  8 24  9  4  1 25  5 10 26 11  3 12 21 13 20  2 14 16 18 28 27 17 23 19 22 29 ...
            29 24  2 16 15 28  1 18 25  7 11 30 14 12  8 17 27 20  4 22  5 23  6  3 19  9 21 26 10 13 ...
            13 18 20 14  9 22 16 21 25 30  2 15 19 10  3 28 24 11  4 12  1  5 26  6  8 23  7 27 29 17 ...
            17 11  1  9  2 18 21 14 19  7  3 26 24  4 25  8 20 15 22 12 27 23 28 16  6 29 13 30 10  5 ...
            5 28 10 17 15 30 29  4  2 23 26  1  3  6  9  7 12 20 27  8 13 25 11 16 14 18 24 22 19 21 ...
            21 27  6 28  2 17 16 22  7  4 14  1 20 29  3 25  9  8 26 30  5 11 10 12 18 13 15 24 19 23 ...
            23 16 28  3  8  5  7 29 21  9  6  1 10 18 26 25 12 17  2 11 15 13 27 22 14 30  4 24 20 19 ...
            19 12 23 11  5 18 29  7 30 20 28  8  1 13  2 10 14 17 21  6 15 25 16 24 26  3 22 27  4  9 ...
            9 23 29  5 14  7 19 17 10 15 18 30 13 26 20 24 21  3 27  1 22 28  6 25  2  8 12 16  4 11 ...
            11  8 27 26  2  6  4 23  1 28 14 25 13 19 18  3 29 15  5 12 30 24  7  9 16 20 10 21 17 22 ...
            22  8  3 30 18  1 21 19 28  5  9 11  2  7 10  4  6 23 12 29 16 13 17 20 25 15 26 14 27 24 ...
            24 23  4 20 26 16 11  6 21  1 27 30 17  3  2  5 15 12 25 14 13 22  9 29 19  8 10  7 28 18 ...
            18  2 27  5 13 16 19  6 11  7 25 28 15  3 20  8 29  1 30 21  4 10 22 17 24 12 26 23  9 14 ...
            14 10 16 23  5 22 11 13 12  2  9 15 27  7  1  4 28 30  8 21 18  6 17 26 19  3 24 29 25 20 ...
            20  9 30  7 22 21 15 10 27  2  1 17  5 24  3 11 18 23  8 25 19 13 28 26 12  4 29 14  6 16 ...
            16 27 18  8 19 29  6 14 20 22 24 28 11 17 25 10  9  3 21 12  5 30 23  2 26  4  7 13  1 15 ...
            15  4 13  8 22 26  5 19 24 17 29  9 27 11 20 23 18 10 25 21  7 14  3  1 16 30  6  2 12 28 ...
            28  4 26 22 10 30 16 25 17 12  6 24 27  9 18  5 20 11 19  2 13  7 15  1  8 14 21 29 23  3 ...
            3  9 13 21 20  1  6 19 14 11 22 30 25 18 12 24  5 16 26  7 23 10  2 28 17  4  8 15 29 27 ...
            27  3 10  8 28 25 23 13 29 22  1  7 16  9 24 15  2 19  5 21 30 11 26 18 14  4 17  6 20 12 ...
            12 22 18  4 19 25 24 30 27 13 23 15  6  5 29 10 20 17  9  1 11 21 28  7 26  8 16  3 14  2 ...
            2 20  5 25 27 12  3 23 30 19 26 28 21  8  7 18 16 29 11 24  1 14 22  4 15  9 17 13  6 10 ...
            10 19 27 14 29  8  4 18 15  7 20 16 12 11 23 21  5  1 26 13  9 25  3 17 30 28 22  2 24  6 ...
            6 18  9 26 17 14  5  3 13  4 27 16  1 12 10 24  8  2 29 28 19 30 22 15 23 20  7 21 11 25 ...
            25 29 26  9 12  7  2 21 16  8 30  3 18 11 27 19 15 14 23 17  1 24 10 28 20  6 22 13  5  4 ...
            4 30 26 29 12 14  8  6 27 21 10 23 22 20 18 17 19 16  7 24 13  3 15 11 28  9  5  2 25  1];
    case 31
        val = [1 ...
            1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 31 ...
            31  1 28  2 18 16 10 19 11 13 25 30 24 15 12  3 14 26  8 20 17 27  4  6 29 21  7 22  5  9 23 ...
            23 25  6 17 30 21 19 27  5 13 18 20 29 26 11 22  7  9  8 10 24 31 14 28 12 15  1  3  2  4 16 ...
            16  9 11 14 25 27  7  2 30 26 10 15 20  3 28  1  5 29 31 17  4 18  8 21 23  6 12 19 24 13 22 ...
            22  8  2 10 26 31 20  5 24  4 14 29 27 23 30 11 28  3 12  1  6 13  7 15 17  9 18 21 16 19 25 ...
            25 18 17 19 12 20 11 21 13 23 28 26 22  9 24  1 27 10  2 29  3 15  5 30 14 16 31  4  7  6  8 ...
            8 11 30 28  4  2 31  5 17 21 24 26 23 13 16  3 29  1 25 14  6  9 12 27 15  7 10 22 20 19 18 ...
            18 10  9 13 11 26  3  1 20 22  2 23 12 28 17  6 30  4  8 24  5 19  7 27 16 14 31 21 15 25 29 ...
            29  2 11  7 12 26  9  3 30  5 31 13 24  6 10  4  1  8 14 17 20 15 18 22 16 21 25 28 19 23 27 ...
            27 29 24 22 18 26 13 19 21 30  1 23 15 28 10 14  8 25 20 31 11  2  5 12 17  7 16  4  3  9  6 ...
            6 18 23 31  2 24 27  8  3 16  7  4 13 10 12 29 11  5 20 30 15 14 19 17 28  9 22 25  1 26 21 ...
            21 26 12  6 16 29  4 11 31  3  7 28  5 22 24 23  1 30 25  2 27 13 20 14 18 15  8 19  9 17 10 ...
            10 25  7 11 17  1 29  5  2  6 28 13  3  8 12 21 27 14 20 16 26 24 18 30  9  4 15 31 23 22 19 ...
            19 26 16  5 15 10 13 12 30  7  3 25 31  6 14  9  1 11 18 24 17  8 22 21  4 20 23 29 28 27  2 ...
            2 25  8 15 11  6  5 10  1  7 13 17  3 31 26 14 21 18 27 30 12 16 28 20  9 19 22 29 23  4 24 ...
            24 12  7 14  4 19 15 22 17 13  6 31 28  8 23 26  1 21 10 29  9 20  2 16 30  3 27 18 11 25  5 ...
            5  7 20  1  9  2 22 26 25 24 30 13  8 27  3 11  4 28  6 15 23 16 18 31 10 21 17 29 19 14 12 ...
            12  8 26  7 31 18 25 15 13 27  1 14  2 28 21  9 29 22  3 10 30  6 23  5  4 17 24 19 16 11 20 ...
            20 18 28 30  8  6 11 27 21 12 22 31  7 23  2 26  4 25  3 24  9 15 19  5 14 13 29 10 16  1 17 ...
            17 31 15 21  5 23 14 22 27  6 19 28  7 24 20  4  9 25 16  2  1 10  3 18 29 12 11  8 13 26 30 ...
            30 16  8 17  2 19 29 18  1 31 22 28 11  9 26 15 24  3 20  6  4 12  5 27 25 13 21 14 10 23  7 ...
            7 29  6 20  8 16 22 30 17 25  4 26 19 10 18  2  9 14 23  3 21 31 27 12 24 11  1 13  5 28 15 ...
            15  6 21 28 31  8 18  3  5 11 23  9 27 17 22  1 12 10  7 25 19 13  4 29 16 24 14 30  2 20 26 ...
            26 20 24 28 14  1  4 30 18  5 16 23 17 11 10 27 31 29 13  2 12  9 21  8  7 19  6 25 22 15  3 ...
            3  6 22  4 10  5 18  7 21 29  8 28 23 11 19 30 27 20 12 25  9 31 24  2 13  1 16 15 26 17 14 ...
            14  3 22  6 24 16 20 25 21  1 18 13  9 30 19  4 23 10  8 31 12  2 15 27 11 29  7 17  5 26 28 ...
            28 16 25 17 12 23 20 27 22 14 24  8 29 15  4 21 11  3 13 31 30 10  6 26  5  1 19  2  7 18  9 ...
            9 16 12 14  5  3 26 18  4 22 10 17 23  8 30 29 20 13 28 24  7  1 15  2 21  6 27 19 31 25 11 ...
            11 15 29 25 10 28 22 13 30 23 18 12 31 19  3 17 16  6  1 24 21 20  7 26  2 14 27  9  5  8  4 ...
            4 27 26  6  2  8  5 21  3 23 19  1 22 11 24 29 17 15  9 28 25 12 18 14  7 30 20 10 31 16 13 ...
            13 15 30 22 12  4 31  9  7  5 25 23 21  2 17 26 29 14 11 16 27 24 10 20 28 18  6  3 19  8  1];
    case 32
        val = [1 ...
            1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 31 32 ...
            32  8 17  3 12 28  1  4 11 14 22 13  5  2 24 10  6  9  7 15 27 16 18 20 19 25 21 23 26 29 31 30 ...
            30 12 11 20 24 21  9 31 10  8 15 13 19 32 14 16 22  2  6 18 17  1 23 25  3  5 26 28 27  7 29  4 ...
            4 32 10  7  3 19  2 14  9  8  1 31  6 15  5 11 21 18 22 12 16 13 17 20 29 23 27 24 26 25 30 28 ...
            28 31  1 27  8 29 19 23 14 17 13 21 20 15 22 30 18 32  6 10  2  4  9  5 24 16 11  3  7 25 12 26 ...
            26 11 31  2  5 29  6 14 32 15 28 30 19 12  3 16 20  7 13 18 23  1 17 21  4 22 24  8 25  9 27 10 ...
            10 22  4 17 25 16  3 18  6  1 19 15 30 21 27 13 32 20  8 26  7  5 28 23 31  9  2 29 24 12 14 11 ...
            11 29 10 19 16 26  2 15 20 17  5 27 21 24 22 28 32  7 18 25  8 30 23  3 31  4  1  6 12  9 14 13 ...
            13  7 19 26 20 16 23 29 25 14  4 18 30  9  3 32 11 15 17 31 24  5  1 21 28 10 12 22 27  2  8  6 ...
            6 11  5  3 17 12 29  1 15 25  4 23 28 24  7 22 26 21 14 19 27 30 32  9 13 20 10 16 31  8  2 18 ...
            18  7  9 23 30 27 15 10 17 32  1 22 29  3 20 31 25 11 19  6 13 16 28 21 26 24  4  8  5 14  2 12 ...
            12  8 10 18  9 19 17 22 31 26 30 11 23 32 21 13 15 24 14 20 25  1 16  7 27 29  2 28  4  3  6  5 ...
            5  8 27 14 18 21  7  4 19  9 15 26 10 28 16 24  3 11 22  6 17 23  2 13 29 12 25 20 30  1 32 31 ...
            31 11 24 30 25 15 29 16 32 23  6 19  1 18 12 27 17 26  3 13 28  2  7 10 20 14  5  4 21  8 22  9 ...
            9  1  8 31 28  3  2 26 12 10 13 22 18 14 30 24 23  4  6 16 25 17  7 32 29 20  5 15 19 21 11 27 ...
            27  9 26 17  6  2 31 12  4  7 14 10 29 18 24 11  8 13  1 20 28 19 22 32  3 15 21 25 23  5 30 16 ...
            16  6 20 12 32 19 24 29 28  5 31  7  1 13  8  3  9  4  2 17 10 30 14 21 15 11 18 26 22 25 27 23 ...
            23 18 27 11 25 13 24 19 14 12 20  9 16 29 17 15  2 30  5 21 10 31 22  1 26  8 32  4 28  7  6  3 ...
            3 14 25 28  6 26  5 32 22  7 11  1 30 29 21 16  2 19 31 13 27  4 10 23 17  8 18 15  9 12 24 20 ...
            20  1 24 15 18 16  4 25 10 32 26  9 21 29 11 28  8  7 23 12 30  2 22  5 13  6 31 17 14  3 27 19 ...
            19  8 16 12  5 17 24  6 23 21 30  7 28  9 11 10 15 31 14 26 32 25 18  2 20  3  1 29 13  4 27 22 ...
            22  8 19 30 26 13 31 29 32  2 25 24 17 27  3 10  4 12  1 28 11  7 16  9 18  5 20 23 15 14  6 21 ...
            21 17 11  2 16 27 20  4 24 28 12 18  8 14 29 26 15  7 31  3 22 10  1  5 19 13 23  9 32 30  6 25 ...
            25 29 27 31 15 32 12  2 23  7 17  4 13  3  8 11  6 22 14 28 18  1  9 30 20 26 16 21 19  5 10 24 ...
            24 27  1  3 21 31 16 10  9 25 22 11  4 14 23  8 12  6 28 20 13 26 19 18 29  5  7  2 32 17 30 15 ...
            15 23 10 14 24 13  2  9 28 25 19 29  7 20 22 21 32 18 31 27  6  8  4 30  3 26  1 11 16  5 12 17 ...
            17 16 14 27 32  5  9  6  4 15 12 31 18 13 25  2 10 21  3 29 22 19 28 26 23 20 11 30  8 24  1  7 ...
            7 30 13  9 22  3 23 11 17 19  4 16  1 10 25 32 24  2 21 12 15  6 29  8 20 27 26 31  5 18 28 14 ...
            14 31 19  7 21  1 12 23 13 30  4 20 32 27  5 16 15  8 28 22 17  2 11 26 18 10  3 25  6 24  9 29 ...
            29 15  1 14  7 12 21  6 27 25  5 22 16  8 23 19  3 30 17  9 24 18 11 32 28 13 10 26  4 31 20  2 ...
            2  1 25 31 21  5 23 22 20  6 30 10 27 18  4 29  9 17 28 15  3 24 32 16 19 11 13 12  7 26 14  8 ...
            8 21  2 27 12 19 10  5 25  7 24 31 23 16 30 22 15  4 26  6 32 13 11  9 20 18  3 28 17 29 14  1]; 
    case 33
        val = [1 ...
            1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 31 32 33 ...
            33 23 11 22 32 27  8  6  9  2 31 17 10 12  3  7 13 24  1 14 30 25 15 18  5  4 16 19 21 20 26 29 28 ...
            28  6 10 18  2  4 11  3 22  9 13 12 14 27 21 31 15 17 23  1 16 20 19 33 30 24 26 32 25  5 29  8  7 ...
            7 12 29  3 27 13 11  1  6 32  2 30 14 25 16 15 19 31 18 17  4  8 20 22 33 24  5 23 21 26 28 10  9 ...
            9 14 10 29  5  7  2 20 12 28 33 16  1 27  6 30  8 11 23 13 25 17 19 15 21 18  3 24  4 32 22 26 31 ...
            31 10 32 28 30  1 17  7 33  8 22  2  5  9  3  6 13 26 20 14 11  4 19 12 15 27 18 16 21 23 25 24 29 ...
            29 25 30 33 20  5 15 24 16 26  4 22 17 27 31  8 10  1 28 18 32 11 19  2 23  3 21  6 12  9  7 14 13 ...
            13  8 32 26 30 15 28  5 25 21 24 14 29 23  4 17  6 11  7 16 22  1 20 27  9 18 12 19  3 31 33  2 10 ...
            10 19 29 26 25  1 30 21 15 31 24  6 17 20 32 12 33 13  5  8 23  9  4 28 27  2 22  3 11 14 16  7 18 ...
            18 26 33 19  4 25 20 29 10 16 27 15  1 21 28 31 22 30 17 32  6 14  2  7  3 23  5 24  8 13  9 12 11 ...
            11 10 13 31  7 32 17 14  8 33  1 15 20 28  3  9 16 12 30 26 18  4 27 19 22 21  2 25 29 24 23  6  5 ...
            5 28 23  2  6 15 12 26  1  7 17 21 29 13  3 30 16  4 24 31  9 19  8 14 20 10 22 11 18 25 27 33 32 ...
            32 18 20 30 22 24 15 14  9 11 21 17 25 23 26  2 27 29 12 31 16 33  3 19 28  7  1  4 10  5 13  6  8 ...
            8  3 15 13  2 24 22 28 32  9 17 11  5 10 23 18 30 19 25 31 14 26 21 16 29 33 27  1 12  4 20  7  6 ...
            6 33 15  3 25 18  1 19  7 20 31 11 16  8 24  2 17  5 26 22 27 23 28 21 30 29  4  9 32 13 10 14 12 ...
            12 16 23 22 25  6 19 18 28 20  8 15 26 24 13 32 10  3 29 11 17  9 27  4 21 33  5  1 31  2 14  7 30 ...
            30 27 14 17  3 18 31 12 32  7 21 25 13  4 15  5 33 26  6 16 28  8  2 29  1 22 10 20  9 23 19 11 24 ...
            24  3 10 33  4  6 18 21 27  5 30  7 23 16 31 13  1  8 12 22 29 19 32 14 28 15 25 11  2 26  9 20 17 ...
            17 29 14 22 20 33 28 12 18 23 27 24 19 30  9 15  2  1 25 32 16  6 21  3 26  8 31  5 11 13  7 10  4 ...
            4  7 15 29 20 11 25 28  2 16 18 27 26  5 21 19 23  8 30 32 24  9 22 31  1 10  6  3 12 17 13 33 14 ...
            14 18 33  9  1  3 32  8 16 10 15 30 12  2 11 20 13 17 24 21  4 23 29 27 22 19  5 31  6 25  7 28 26 ...
            26 14 19  9 28 22  4 12 20 15 10 30 23 31  3 33  6 24 11 27  7 25  2 32  1  5 16 13 18 29 17  8 21 ...
            21 32 23 30 10 31  4 18 13 27  3 28 19 14 24 33 17  1 11  6  2  8  5 20 25  9 26 12  7 29 15 22 16 ...
            16 14 21  1  9  6 20 23  7 11 26  3  2 12  8 25 22  5 27 17 28  4 29 31 30 13 15 33 18 10 24 32 19 ...
            19 10 25  3  1 13 16 24  7  4 26 11 28  9 29 21  5 32 15  6 22  8 27 12 23 17 30  2 18 14 33 31 20 ...
            20  1 18 24 27 10 21  7 19 13 22  6 23 32  5  2  9 30  3  8 28 14 31 26 17 12 25  4 33 29 16 11 15 ...
            15 32  3  5 17 16 25 33 21  8 18 11  9 24 10  2 13 20  4 30 28  1 26 19  6 31 29  7 22 14 23 12 27 ...
            27 16 30 20  6 26  7  9 21 10  8 29 18 15 23 14 32  4  1 33 22 12  5  3 13 19 24 17  2 28 11 31 25 ...
            25  8 17 31 19 26 10 27 32 20 24 18  6 28 13 23 15  7  5 14  3 16  9 33 11 30  4  2 21 12  1 29 22 ...
            22 13 21  9 25 19 16  5 12 24 20  2 15 11 29  6  4  3 14  1 32 31 28 17 33 10  7 27 30 18  8 26 23 ...
            23 10 17 22 15  4 13 30 11  8 19  1 24 28 25 12  6 29  2 33  7 26 16 32 21 14  5 18  9 31 27 20  3 ...
            3 20 18 22  7 24 30  6 27 25 14  4 31 21 11 32 29  9  5 19 17 15  8  1 23 33 12 10 26 13 28 16  2 ...
            2 19 27 11 33 25 10 28 24 12 21 13 29 32 30  5 22 18  7 31 23 20 16  3 17 26 15  9  8  4 14  6  1];
    case 34
        val = [1 ...
            1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 31 32 33 34 ...
            34 20  7  9 11 23 29 21 24 28 14 16 31 15 30 17 32 27 10 33  3  5  1 25 18  8  2 19 12 22 13 26  4  6 ...
            6 18 23 16  2 17 26 28  3 12  5 19 27 20  1 29 24  4 25 15 13  7 30 33 21 11 22 34 31  8 10 32 14  9 ...
            9 22 15 25  4 11 13 10 27 33  5 26  6 19  8 20 34 23 31 30 32 24 12 14 18 28  1  3 21  2  7 16 29 17 ...
            17 31 22  5 16  4 18 11 24 34  1 32 23 19 25 33 30 12 20  2 13  8 28  6  3 14 29  7 10 26 21  9 15 27 ...
            27 14 21 32 25 11 33  8 13 28 18 12 34  2  9 26  3 15 22 24 29 31 10 30  5 17 16  1 20  4 19  6 23  7 ...
            7  6 17 34 11 25 19 10  4 24 33 23 30 26  2 32 28 21  5  3  1  8 29 14 20 31 12  9 13 15 18 16 27 22 ...
            22 21 29  1 19 30 14 23 18 31  4 33 15 34 12 11 32 26  5 20  3 24 16 13 17 25  2 27  6  9  7 28 10  8 ...
            8 30 13  5  9 34  7  2 10 24  1 31 17 11 20 22  3 16  6 32  4 12 15 19 26 14 33 27 18 21 23 25 29 28 ...
            28  2  4 20 12 26 13  9  8 16 10 14 17 15 21 18 29  6 22 31  7 32  5 23 27 19 11  1 30 24  3 25 34 33 ...
            33 32  1 27  5  7 17 22 10 13 11 16 23  6 12  2  8 15  3 18 24  9 25 28 19 21 30 20 26 29  4 31 34 14 ...
            14  7  3 11 29 22  2 23  1 33 12  6 24 17 27  4 21 13 30  8 18 32  9 28 15  5 10 16 19 31 26 34 25 20 ...
            20 27  2 33 28 24  8 31  9 23 21 34  3 26 10  1 22 16  7 15 29  5  4 30 25 12 17 13 32  6 11 14 19 18 ...
            18 10 19  9  4 22 30 15 14 28 16 11 17 20  5  8 23 26 24 21 25 31 33 29 27 32  3 34  6  1  7 13  2 12 ...
            12 10 23 28 13  6 15  1  4  7 21  8 22 11 30 34 32  2  5 14 24 18 20 29 16 25 17  9 27 26 19 33 31  3 ...
            3 13 23  5 25 16 32 31  1 34 26 15 24 19 28 17 14  2 30 27  7 29 11  4  8 21  6 20  9 12 18 22 33 10 ...
            10 28 33 19 16 12 21 17 23  9 31  5 11 26 18  6  4 27  8 32 20 30 22  1 24 13 25 14  3  2 15  7 34 29 ...
            29 32 10 17 30  6 33 25  1 11 34 13 12 19 22  4 23  2 14 31 16  9 20 15 26  7 24  5 18  3 28  8 27 21 ...
            21 26 30 16 18  1 28 22 29 33  2 20 17  3 27 25 32  7 23 34  4  9  5 24 10  6 31 11 15 12  8 14 13 19 ...
            19 29 23 22 18 33  4 10 20 14 25 30 21 27 24 26 31  6 13 34 17 28 32  8  1  5  2 11  7 12  3  9 16 15 ...
            15  9 29 12 27 16  3  6 14 22 17 24 11 31 28 23 13 33  1 18 25  5 21  4 26 32 19  7 20  8 34 30 10  2 ...
            2 16 30  1 13 27  9 24 31 14  5 28 26 33 18 15 17  7  4 29  3 32 21 10 34 19 23 12 25 22 20  6  8 11 ...
            11 18 13 16  5 12 28  4 14 30 19 32 15 20 10  3 17  8  7 22  6 29 25  9 21 31  2 34 24 27  1 23 33 26 ...
            26 25  3 19 13 18  9 30 23 32 34 21 28 20 33  6 16  8 12 31 24 14 27 11 10 29 15  4  1 17  2 22  7  5 ...
            5 29 34  9 32 11 19 24 20 25 21  3  7 14  1 26  8 17 33 13 31 18 30 28 27 15  2  6 10 22 12 23  4 16 ...
            16 20 11 27 29  2 25 24 15  6 21 33  7 31 19 14 34 18 17  1  9  3  8  4 28 12 32 13 22 26 23 10  5 30 ...
            30 29  8 24  2 18 27 17  4 13  1 21 14 32 22  9 19 15 11 28 34 10 12  7 25  6  5 33 16 26 20 23  3 31 ...
            31 13 29 26  1  6 25 23 15 32 30  7 11  2 21 12 16 22 27  3 10 18 14  8 19  4 17  5 34 28  9 33 20 24 ...
            24  6  2 29  9 14  4 15 23  8 26 11  3 20 32 12 30 18 34 16 21  1 10 25  7 33 17 19  5 22 28 31 27 13 ...
            13 21 15 10 31 20 16 24  7 18 26 22  8 25 27 23 11  6 28 30  2  1 14 12  4  3 33  9 17 29 19 34  5 32 ...
            32 16 28 25  8 33 22 14 26  9 18  2 24 23 17  6 34 27 12 29 10  7  1 15 31 21 20 19  3 30 11  5 13  4 ...
            4 32 17 10 21 19  1 12 24 30  9  2 28  7 27 31 29 20 18  5 15 33 14 11  8  6 26 16 34 22 25 13  3 23 ...
            23 14 10  9  6 27 30  3 29 13 20 28 11 21  7 19  2 26 17 12  1 16 33 24 22 32 18  4 34 15  8  5 31 25 ...
            25 10 15 28  5 27 34  8  3 22 19 17 21 16 14  6 30  4  2 31 23 20 13 24 32 29 18  7 26 12 33 11  9  1];
    otherwise
        error('continuous carryover sequence defined only for 2, or 6-34 treatments');

end
