function t = mcsInitFunc(t,varargin)
%function t =  mcsInitFunc(t,varargin)
%
%
% this is the initFunc for MCS trackers. The idea is to arrange an array
% with all combinations of a number of tracked variables. Be sure to set
% t.params with the following fields:
%
%   tracked_vars: a struct. each field should have a name indicating the
%       tracked variable name (e.g. freq, amp, or ITD). It's value should
%       be an array with the unique values to be tested. You can tweak the
%       distribution of values by entering more than one copy of a
%       particular value...
%   numtrials_perstim: a number. tracker will test each combination of
%       values given in tracked_vars this many times.
%   shuf_rule: a string indicating rule to govern shuffling of trials
%       'full' means to shuffle the entire set of trials
%       'blocks' means to shuffle so that all combos are presented once
%           before proceeding to the second presentation
%       'cont_carryover' - continuous carryover sequence
%       'none' 'fixed_sequence' - do not shuffle; repeat sequence of trials
%           on each block
%       'fixed_blocks' - do not shuffle; complete all of one value
%           before proceeding to next
%
% optionally, t.params may also contain the fields:
%    special_stim: a struct array with fields matching tracked_vars. each
%       element of the struct array defines a specific combination of
%       variables that will be added to the set of stim generated by all
%       combinations of the tracked_vars. Use this for example to insert a
%       "silent" block without causing the tracker to pair "silence" with all
%       values of other combinations, or to add extra "conditioning" or
%       "adaptation" trials to a randomly generated test set. 

% load in the tracker parameters
params = get(t,'params');

% I'm guessing that params should come in with some parameter values in
% arrays

% initialize the tracker status
status = struct('currentTrial',1);
status.done = 0;

t = set(t,'status',status);


% set up the tracker trialdata 
%
% include all the fields given in t.params.tracked_vars, as well as
% response and target. 


% first generate a matrix with all stim combinations
allcombos = [];

% make one copy of each stim combo
varnames = fieldnames(params.tracked_vars);  % {'freq' 'amp' 'ITD'}
for iVar = 1:length(varnames)
    varname = varnames{iVar}; % 'freq' then 'amp'
    varvals = params.tracked_vars.(varname); % [100 200 300] then [20 40]
    if iscell(allcombos) && ~iscell(varvals) % CS 11/4/2015 to allow cell inputs 
        varvals = num2cell(varvals);
    elseif iscell(varvals) && ~iscell(allcombos)
        allcombos = num2cell(allcombos);
    end
        if ~isempty(varvals)
        orig_length = size(allcombos,2);
        if iVar == 1
            allcombos = varvals;
        else
            allcombos = repmat(allcombos,1,length(varvals)); % [] then [100 200 300 100 200 300]
            allcombos(iVar,:) = reshape(repmat(varvals,orig_length,1),1,[]);
            % [100 200 300] then [100 200 300 100 200 300; 20 20 20 30 30 30]
        end
    end
end


% next turn the matrix into an appropriate struct array
if ~iscell(allcombos)
    foo = num2cell(allcombos);  % CS 11/04/2015
else 
    foo = allcombos;
end
trialdata_1 = cell2struct(foo,varnames)'; % now 1xnumtrials struct

% CS 8/31/2010: add a way to insert a special stim combo (i.e. one that is
% not generated by complete combination of tracked variables. 
if isfield(params,'special_stim')
    trialdata_1 = [trialdata_1 params.special_stim];
end


trialdata = trialdata_1(1:0); % empty struct array with correct fields

switch(params.shuf_rule)
    case {'full' 'blocks'}
        % now assemble the full trialdata by combining numtrials_perstim copies of
        % trialdata_1
        for iTrial = 1:params.numtrials_perstim
            % shuffle each set
            trialdata = [trialdata trialdata_1(randperm(length(trialdata_1)))];
        end

        % if full shuffle, shuffle the full array
        if strcmp(params.shuf_rule,'full')
            trialdata = trialdata(randperm(length(trialdata)));
        end
    case {'cont_carryover'}
        fprintf('Continuous Carryover for mcs tracker is currently under test. No guarantees yet.\n');
        
        fprintf('Note that parameter <numtrials_perstim> is interpreted to indicate the number of complete passes through the carryover sequence.\n');
        
        % the idea here is to use the carryover sequence for the number of
        % combos as an index into the specific stim combos stored in
        % trialdata_1
        
        % first compute the carryover sequence for numcombos
        cc_seq = cont_carryover_seq(length(trialdata_1));
        fprintf('Generating continuous carryover sequence for %d treatments. Total trials: [%d x %d] = %d\n',...
            length(trialdata_1),length(cc_seq),params.numtrials_perstim,(length(cc_seq)-1)*params.numtrials_perstim+1);
        
        % randomly assign combos to treatment numbers by shuffling the
        % array of combos.
        trialdata_1 = trialdata_1(randperm(length(trialdata_1)));
        
        
        % generate the sequence of trial structures. do not shuffle.
        for i = 1:params.numtrials_perstim
            if i==1
                trialdata = trialdata_1(cc_seq); % first pass through the sequence prepends a 1 to start with pair [1 1]
            else
                trialdata = [trialdata trialdata_1(cc_seq(2:end))]; % note not to use the initial "1" for the 2nd and later passes.
            end
        end
    
    case {'none' 'fixed_sequence'}  % do not shuffle at all % CS 04/28/17
         % now assemble the full trialdata by combining numtrials_perstim copies of
        % trialdata_1
        for iTrial = 1:params.numtrials_perstim
            % shuffle each set
            trialdata = [trialdata trialdata_1];
        end
        
    case {'fixed_blocks'}  % do not shuffle at all % CS 04/28/17
         % now assemble the full trialdata by combining numtrials_perstim copies of
        % trialdata_1
        for iTrial = 1:params.numtrials_perstim
            % shuffle each set
            trialdata = [trialdata' trialdata_1']';
        end
        trialdata = reshape(trialdata,1,[]);
end

% add the response fields

trialdata(1).response = [];
trialdata(1).target = [];
trialdata(1).correct =  [];

% now inset the trialdata into t
t = set(t,'trialdata',trialdata);